/************************************************************************************************************\

Module Name:    CopyModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "CopyModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------
static const    MCHAR8      g_szModuleNameBase[]    = "Copy";
static          MUINT32     g_uiCopyModCount        = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       CopyMod_Init

Description:    .

\************************************************************************************************************/
LStatus CopyMod_Init(
            CopyModule*         poCopyMod,
            LDevice_Handle      hDevice,
            MUINT               uiOutBufferCount,
            LBuffer_Attributes* poOutBufferAttributes)
{
    MsgLog(2, "{...");

    CopyMod_Cleanup(poCopyMod);

    LStatus eStatus = ((poCopyMod != MNULL)
                       && (hDevice != MNULL)
                       && (poOutBufferAttributes != MNULL))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "copy"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poCopyMod->hDevThread));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBlit_CreateAttributes oAttributes = {LBlit_AttributeType_CREATE, MTRUE, MTRUE};

        eStatus = LBlit_Create(hDevice, poCopyMod->hDevThread, &oAttributes.eType, &(poCopyMod->hBlit));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(poCopyMod->szModuleName,
                 sizeof(poCopyMod->szModuleName),
                 "%s%d",
                 g_szModuleNameBase,
                 g_uiCopyModCount);

        eStatus = ModLnk_Init(
                        &(poCopyMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        poOutBufferAttributes,
                        MFALSE,
                        0,
                        poCopyMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poCopyMod->oSrcPos.iX       = 0;
        poCopyMod->oSrcPos.iY       = 0;
        poCopyMod->oDstRect.iLeft   = 0;
        poCopyMod->oDstRect.iTop    = 0;

        switch (poOutBufferAttributes->eAttributeType)
        {
            case LBuffer_Type_VIDEO:
                poCopyMod->oDstRect.iRight  = poOutBufferAttributes->oVideoAttributes.uiWidth;
                poCopyMod->oDstRect.iBottom = poOutBufferAttributes->oVideoAttributes.uiHeight;
                poCopyMod->bLinear          = MFALSE;
                break;

            case LBuffer_Type_SYSTEM_VIDEO:
                poCopyMod->oDstRect.iRight  = poOutBufferAttributes->oSystemVideoAttributes.uiWidth;
                poCopyMod->oDstRect.iBottom = poOutBufferAttributes->oSystemVideoAttributes.uiHeight;
                poCopyMod->bLinear          = MFALSE;
                break;

            case LBuffer_Type_LINEAR:
                poCopyMod->oDstRect.iRight  = poOutBufferAttributes->oLinearAttributes.uiSize;
                poCopyMod->oDstRect.iBottom = 1;
                poCopyMod->bLinear          = MTRUE;
                break;

            case LBuffer_Type_SYSTEM_LINEAR:
                poCopyMod->oDstRect.iRight  = poOutBufferAttributes->oSystemLinearAttributes.uiSize;
                poCopyMod->oDstRect.iBottom = 1;
                poCopyMod->bLinear          = MTRUE;
                break;

            default:
                eStatus = LStatus_INVALID_PARAM;
                break;
        }
    }
    else
    {
        CopyMod_Cleanup(poCopyMod);
    }

    ++g_uiCopyModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       CopyMod_Cleanup

Description:    .

\************************************************************************************************************/
void CopyMod_Cleanup(CopyModule* poCopyMod)
{
    MsgLog(2, "{...");

    if (poCopyMod != MNULL)
    {
        ModLnk_Cleanup(&(poCopyMod->oOutLink));

        if (poCopyMod->hBlit != MNULL)
        {
            while (LSTATUS_IS_FAIL(LBlit_Destroy(poCopyMod->hBlit)))
            {
                MsgLog(0, "LBlit_Destroy failed! Retry...");
                usleep(1000*1000);
            }

        }

        if (poCopyMod->hDevThread != MNULL)
        {
            LDeviceThread_Destroy(poCopyMod->hDevThread);
        }

        poCopyMod->hBlit            = MNULL;
        poCopyMod->hDevThread       = MNULL;
        poCopyMod->oInLink.poModLnk = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       CopyMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus CopyMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    CopyModule* poCopyMod   = (CopyModule*)pvData;
    BufferInfo* poDstBuffer = MNULL;
    BufferInfo* poSrcBuffer = MNULL;

    ModThread_SetName(poCopyMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poCopyMod->oCpuThread.bKillThread)
    {
        LStatus eStatus = LStatus_OK;

        if (poDstBuffer == MNULL)
        {
            eStatus = ModLnk_GetReturnedBuffer(
                        &(poCopyMod->oOutLink),
                        100,
                        poCopyMod->hDevThread,
                        &poDstBuffer);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                        &(poCopyMod->oInLink),
                        100,
                        0,
                        poCopyMod->hDevThread,
                        &poSrcBuffer,
                        MNULL,
                        MNULL);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poDstBuffer->bEndOfStream = MTRUE;
                poCopyMod->oCpuThread.bKillThread = MTRUE;
            }
            else
            {
                if (poCopyMod->bLinear)
                {
                    if (poSrcBuffer->uiSizeBytes > 0)
                    {
                        MsgLog(4, "LBlit_Copy(DstBuffer[%u], SrcBuffer[%u]: Offset= %u, Size= %u)...",
                                  poDstBuffer->uiId, poSrcBuffer->uiId, poSrcBuffer->uiStartOffset,
                                  poSrcBuffer->uiSizeBytes);

                        // Align on dword for better performance.
                        MUINT uiLeftPad = poSrcBuffer->uiStartOffset & 3;

                        eStatus = LBlit_Copy(
                                    poCopyMod->hBlit,
                                    poSrcBuffer->hBuffer,
                                    poDstBuffer->hBuffer,
                                    poSrcBuffer->uiStartOffset - uiLeftPad,
                                    0,
                                    (poSrcBuffer->uiSizeBytes + uiLeftPad + 3) & ~3);

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            poDstBuffer->uiStartOffset  = uiLeftPad;
                            poDstBuffer->uiSizeBytes    = poSrcBuffer->uiSizeBytes;

                            MsgLog(
                                4,
                                "LBlit_Copy done. Dst: Offset= %u, Size= %u (status= %d - %s)",
                                poDstBuffer->uiStartOffset,
                                poDstBuffer->uiSizeBytes,
                                eStatus,
                                GetStatusStr(eStatus));
                        }
                        else
                        {
                            MsgLogErr(
                                "ERROR! LBlit_Copy returned status= %d (%s).",
                                eStatus,
                                GetStatusStr(eStatus));
                        }
                    }
                }
                else
                {
                    MsgLog(4, "LBlit_Blit(DstBuffer[%u], SrcBuffer[%u])...",
                              poDstBuffer->uiId, poSrcBuffer->uiId);

                    eStatus = LBlit_Blit(
                                poCopyMod->hBlit,
                                poSrcBuffer->hBuffer,
                                poDstBuffer->hBuffer,
                                &(poCopyMod->oSrcPos),
                                &(poCopyMod->oDstRect),
                                MNULL,
                                LBlit_Bop_S,
                                MNULL,
                                MNULL);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MsgLog(4, "LBlit_Blit done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));
                    }
                    else
                    {
                        MsgLogErr(
                            "ERROR! LBlit_Blit returned status= %d (%s).",
                            eStatus,
                            GetStatusStr(eStatus));
                    }
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                poDstBuffer->uiTimestampUsec = poSrcBuffer->uiTimestampUsec;
                poDstBuffer->uiSyncPtsUsec = poSrcBuffer->uiSyncPtsUsec;
                ModLnk_SubmitBuffer(&(poCopyMod->oOutLink), poDstBuffer, poCopyMod->hDevThread, NO_TAG);
                ModLnkIn_ReturnBuffer(&(poCopyMod->oInLink), poSrcBuffer, poCopyMod->hDevThread, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poCopyMod->oOutLink), poDstBuffer);
                ModLnkIn_ReturnBuffer(&(poCopyMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
            }

            poDstBuffer = MNULL;
            poSrcBuffer = MNULL;
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (poDstBuffer != MNULL)
    {
        ModLnk_ReleaseBuffer(&(poCopyMod->oOutLink), poDstBuffer);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       CopyMod_Start

Description:    .

\************************************************************************************************************/
LStatus CopyMod_Start(CopyModule* poCopyMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poCopyMod != MNULL)
        && (poCopyMod->hBlit != MNULL))
    {
        if ((poCopyMod->oInLink.poModLnk != MNULL)
            && (poCopyMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = ModThread_Start(&(poCopyMod->oCpuThread), poCopyMod, CopyMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       CopyMod_Stop

Description:    .

\************************************************************************************************************/
void CopyMod_Stop(CopyModule* poCopyMod)
{
    MsgLog(2, "{...");

    if (poCopyMod != MNULL)
    {
        ModThread_Stop(&(poCopyMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

